import React, { useState } from 'react';
import { DownloadIcon } from './icons/DownloadIcon';

interface GenerationResultProps {
  files: string[];
  onDownload: () => void;
}

type BuildTarget = 'apk' | 'aab';

const StyleInjector: React.FC = () => (
    <style>{`
      @keyframes fadeIn {
        from { opacity: 0; transform: translateY(10px); }
        to { opacity: 1; transform: translateY(0); }
      }
      .animate-fade-in {
        animation: fadeIn 0.5s ease-out forwards;
      }
    `}</style>
  );

export const GenerationResult: React.FC<GenerationResultProps> = ({ files, onDownload }) => {
  const [activeTab, setActiveTab] = useState<BuildTarget>('aab');
  const [showFiles, setShowFiles] = useState(false);

  const tabBaseClasses = "px-4 py-2 text-sm font-medium rounded-md transition-colors duration-200 focus:outline-none focus:ring-2 focus:ring-offset-2 focus:ring-offset-slate-800 focus:ring-indigo-500";
  const activeTabClasses = "bg-indigo-600 text-white";
  const inactiveTabClasses = "bg-slate-700 text-slate-300 hover:bg-slate-600";

  const renderContent = () => {
    if (activeTab === 'aab') {
      return (
        <div className="space-y-4 animate-fade-in">
            <h4 className="font-semibold text-white">Build an App Bundle (for Google Play)</h4>
            <ol className="list-decimal list-inside text-sm text-slate-400 space-y-2">
                <li>Download and unzip the Android Studio project.</li>
                <li>In Android Studio, go to <code className="text-xs bg-slate-900/50 p-1 rounded">Build &gt; Generate Signed Bundle / APK...</code></li>
                <li>Select 'Android App Bundle', then create a new signing key.</li>
                <li>Build the bundle. The <code className="text-xs bg-slate-900/50 p-1 rounded">.aab</code> file is what you upload to the Play Store.</li>
            </ol>
             <p className="text-xs text-slate-500 mt-3">Full step-by-step instructions are in the README.md file inside the zip.</p>
        </div>
      );
    }
    if (activeTab === 'apk') {
      return (
        <div className="space-y-4 animate-fade-in">
            <h4 className="font-semibold text-white">Build an APK (for Testing)</h4>
            <ol className="list-decimal list-inside text-sm text-slate-400 space-y-2">
                <li>Download and unzip the Android Studio project.</li>
                <li>In Android Studio, go to <code className="text-xs bg-slate-900/50 p-1 rounded">Build &gt; Build Bundle(s) / APK(s) &gt; Build APK(s)</code>.</li>
                <li>Android Studio will build a debug APK.</li>
                <li>Click the 'locate' link in the notification to find the <code className="text-xs bg-slate-900/50 p-1 rounded">.apk</code> file for direct installation.</li>
            </ol>
            <p className="text-xs text-slate-500 mt-3">Full step-by-step instructions are in the README.md file inside the zip.</p>
        </div>
      );
    }
    return null;
  }

  return (
    <>
    <StyleInjector />
    <div className="mt-8 pt-6 border-t border-slate-700 animate-fade-in">
      <h3 className="text-lg font-semibold text-green-400">Generation Complete!</h3>
      <p className="text-sm text-slate-400 mt-1 mb-4">Your Android project is ready. Choose your build target for instructions.</p>
      
      <div className="flex space-x-2 bg-slate-800 p-1 rounded-lg mb-4">
        <button 
            onClick={() => setActiveTab('aab')}
            className={`${tabBaseClasses} ${activeTab === 'aab' ? activeTabClasses : inactiveTabClasses} w-1/2`}
        >
            App Bundle (AAB)
        </button>
        <button 
            onClick={() => setActiveTab('apk')}
            className={`${tabBaseClasses} ${activeTab === 'apk' ? activeTabClasses : inactiveTabClasses} w-1/2`}
        >
            APK
        </button>
      </div>

      <div className="my-4 p-4 bg-slate-900/50 rounded-lg border border-slate-700 min-h-[200px] flex flex-col justify-center">
        {renderContent()}
      </div>

      <button
        onClick={onDownload}
        className="w-full bg-green-600 hover:bg-green-500 text-white font-bold py-3 px-4 rounded-lg text-lg transition-all duration-300 transform hover:scale-105 flex items-center justify-center shadow-lg"
      >
        <DownloadIcon className="w-6 h-6 mr-2" />
        Download Project (.zip)
      </button>

      <div className="mt-4 text-center">
        <button onClick={() => setShowFiles(!showFiles)} className="text-sm text-slate-400 hover:text-white transition-colors">
          {showFiles ? 'Hide' : 'Show'} generated file list
        </button>
      </div>

      {showFiles && (
        <div className="mt-2 p-4 bg-slate-900/50 rounded-lg border border-slate-700 max-h-48 overflow-y-auto animate-fade-in">
          <ul className="text-xs font-mono text-slate-300 space-y-1">
            {files.map((file, index) => (
              <li key={index} className="truncate">- {file}</li>
            ))}
          </ul>
        </div>
      )}
    </div>
    </>
  );
};