import { GoogleGenAI } from "@google/genai";

const API_KEY = process.env.API_KEY;
if (!API_KEY) {
    throw new Error("API_KEY environment variable not set");
}

const ai = new GoogleGenAI({ apiKey: API_KEY });

const createPrompt = (websiteUrl: string, appName: string): string => {
  return `
You are an expert Android developer specializing in creating clean, modern, and functional applications. Your task is to generate the complete source code for a simple Android application. The application's main purpose is to display a specific website in a fullscreen WebView.

The application must first display a splash screen and then navigate to the main activity containing the WebView.

**Application Requirements:**
1.  **Application Name:** The name of the application should be \`${appName}\`. This should be set in \`app/src/main/res/values/strings.xml\` and referenced in \`app/src/main/AndroidManifest.xml\` as the application label.
2.  **Splash Screen:** An initial activity that displays a splash screen image. The image file will be named \`splash_screen.png\` and will be located in the \`app/src/main/res/drawable/\` directory.
3.  **Main Activity:** After a short delay (e.g., 2 seconds), the splash screen should transition to the main activity.
4.  **WebView:** The main activity will contain a WebView that loads and displays the following website: \`${websiteUrl}\`.
5.  **WebView Configuration:**
    - The WebView should fill the entire screen.
    - JavaScript must be enabled.
    - The app should handle basic page navigation within the WebView. Pressing the back button should navigate to the previous page in the WebView's history if possible.
6.  **Permissions:** The \`AndroidManifest.xml\` must include the \`INTERNET\` permission.
7.  **Modern Android Standards:** Use recent \`targetSdkVersion\` and \`compileSdkVersion\` (e.g., 34). Use AndroidX libraries.

**Output Format:**
The output MUST be a single, valid JSON object. The keys of this object should be the full file paths (e.g., 'app/src/main/AndroidManifest.xml'), and the values should be the complete code content for each file as a string.

**Required Files:**
1.  \`app/src/main/AndroidManifest.xml\`
2.  \`app/src/main/res/layout/activity_main.xml\` (For the WebView)
3.  \`app/src/main/res/layout/activity_splash.xml\` (For the Splash Screen Image)
4.  \`app/src/main/res/values/styles.xml\` (To define the fullscreen splash screen theme)
5.  \`app/src/main/java/com/example/webviewapp/MainActivity.java\`
6.  \`app/src/main/java/com/example/webviewapp/SplashActivity.java\`
7.  \`app/build.gradle\` (Module: app)
8.  \`app/src/main/res/values/colors.xml\`
9.  \`app/src/main/res/values/strings.xml\` (Ensure it contains an entry for 'app_name' set to "${appName}")

Please generate the content for all the files listed above.
`;
};

export const generateAppCode = async (websiteUrl: string, appName: string): Promise<Record<string, string>> => {
  const prompt = createPrompt(websiteUrl, appName);

  try {
    const response = await ai.models.generateContent({
      model: "gemini-2.5-pro",
      contents: prompt,
      config: {
        responseMimeType: "application/json",
        temperature: 0.1,
      },
    });

    const jsonString = response.text;
    // The response might be wrapped in markdown backticks
    const cleanedJsonString = jsonString.replace(/^```json\s*|```\s*$/g, '');
    const files = JSON.parse(cleanedJsonString);
    return files;

  } catch (error) {
    console.error("Error calling Gemini API:", error);
    throw new Error("Failed to generate code via Gemini API.");
  }
};